import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import Link from "next/link";
import { getSessionOptional } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { transferLefaToLefa } from "@/lib/ledger";
import { GuestPay } from "@/components/guest/GuestPay";

export default async function TextPayTokenPage({ params }: { params: { token: string } }) {
  const session = await getSessionOptional();

  const reqRow = await prisma.textPayRequest.findUnique({
    where: { token: params.token },
    include: { fromUser: true },
  });

  async function payWithLefa() {
    "use server";
    if (!reqRow) throw new Error("NOT_FOUND");
    if (reqRow.status === "PAID") return reqRow.paidTxId ?? "";
    if (!session) throw new Error("UNAUTHENTICATED");

    const tx = await transferLefaToLefa({
      fromUserId: session.uid,
      toEmail: reqRow.fromUser.email,
      amountMinor: reqRow.amount,
      description: reqRow.description ?? `Text-to-pay${reqRow.reference ? ` ${reqRow.reference}` : ""}`,
    });

    await prisma.textPayRequest.update({
      where: { id: reqRow.id },
      data: { status: "PAID", paidTxId: tx.transaction.id },
    });

    return tx.transaction.id;
  }

  if (!reqRow) {
    return (
      <>
        <TopNav />
        <Container>
          <div className="mx-auto max-w-xl">
            <Card>
              <CardHeader title="Payment request" subtitle="Not found" />
              <CardBody>
                <div className="text-sm text-zinc-700">This link is invalid or expired.</div>
              </CardBody>
            </Card>
          </div>
        </Container>
      </>
    );
  }

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="Text-to-Pay" subtitle={reqRow.status === "PAID" ? "Completed" : "Confirm payment"} />
            <CardBody>
              <div className="rounded-xl border border-zinc-200 p-4">
                <div className="text-xs text-zinc-600">Pay to</div>
                <div className="text-sm font-medium">{reqRow.fromUser.email}</div>

                <div className="mt-2 text-xs text-zinc-600">Amount</div>
                <div className="text-2xl font-semibold">P {fromMinor(reqRow.amount)}</div>

                {reqRow.reference ? <div className="mt-2 text-xs text-zinc-600">Reference: {reqRow.reference}</div> : null}
                {reqRow.description ? <div className="mt-1 text-xs text-zinc-600">{reqRow.description}</div> : null}
                <div className="mt-3 text-xs text-zinc-500">Status: {reqRow.status}</div>
              </div>

              {reqRow.status === "PAID" ? (
                <div className="mt-4 text-sm text-green-700">Payment completed.</div>
              ) : (
                <>
                  {session ? (
                    <form action={payWithLefa} className="mt-4">
                      <Button type="submit" className="w-full">Pay with LEFA</Button>
                    </form>
                  ) : (
                    <div className="mt-4 grid gap-2">
                      <Link href={`/signin?callbackUrl=/t/pay/${reqRow.token}`}>
                        <Button className="w-full">Sign in to pay with LEFA</Button>
                      </Link>
                      <GuestPay mode="TEXT_PAY" token={reqRow.token} />
                    </div>
                  )}
                </>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
